"use client";
import { useEffect, useState, FormEvent } from "react";
import { useSession, signOut } from "next-auth/react";
import Link from "next/link";
import { useLanguage } from "@/contexts/LanguageContext";
import {
  ArrowLeft,
  LogOut,
  Save,
  AlertCircle,
  CheckCircle2,
} from "lucide-react";

export default function ProfilePage() {
  const { t } = useLanguage();
  const { status } = useSession();

  const [loading, setLoading] = useState(true);
  const [err, setErr] = useState<string | null>(null);
  const [msg, setMsg] = useState<string | null>(null);

  const [name, setName] = useState("");
  const [email, setEmail] = useState("");

  // password form
  const [currentPassword, setCurrentPassword] = useState("");
  const [newPassword, setNewPassword] = useState("");
  const [savingProfile, setSavingProfile] = useState(false);
  const [savingPassword, setSavingPassword] = useState(false);

  useEffect(() => {
    if (status === "unauthenticated") {
      setLoading(false);
      setErr(t("profile.mustLogin") || "You must log in to view your profile.");
      return;
    }
    if (status === "authenticated") {
      fetch("/api/me")
        .then((r) => r.json())
        .then((j) => {
          if (!j.ok) throw new Error(j.error || "failed");
          setName(j.user?.name ?? "");
          setEmail(j.user?.email ?? "");
        })
        .catch((e) => setErr(String(e)))
        .finally(() => setLoading(false));
    }
  }, [status]);

  const onSaveProfile = async (e: FormEvent) => {
    e.preventDefault();
    setErr(null);
    setMsg(null);
    setSavingProfile(true);
    try {
      const resp = await fetch("/api/me", {
        method: "PATCH",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ name, email }),
      });
      const json = await resp.json();
      if (!json.ok) throw new Error(json.error || "Update failed");
      setMsg(
        t("profile.updateSuccess") ||
          "Profile updated. If you changed email, you may need to log out/in to refresh the session."
      );
    } catch (e: any) {
      setErr(String(e?.message || e));
    } finally {
      setSavingProfile(false);
    }
  };

  const onChangePassword = async (e: FormEvent) => {
    e.preventDefault();
    setErr(null);
    setMsg(null);
    setSavingPassword(true);
    try {
      const resp = await fetch("/api/me/password", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ currentPassword, newPassword }),
      });
      const json = await resp.json();
      if (!json.ok) throw new Error(json.error || "Password change failed");
      setMsg(t("profile.passwordUpdated") || "Password updated.");
      setCurrentPassword("");
      setNewPassword("");
    } catch (e: any) {
      setErr(String(e?.message || e));
    } finally {
      setSavingPassword(false);
    }
  };

  if (loading)
    return (
      <main className='max-w-3xl mx-auto px-4 py-6'>
        {t("common.loading") || "Loading…"}
      </main>
    );

  if (err && status !== "authenticated") {
    return (
      <main className='max-w-xl mx-auto px-4 py-16'>
        <div className='alert alert-error mb-4'>
          <AlertCircle />
          <div>
            <p className='font-semibold'>
              {t("profile.accessRequired") || "Access required"}
            </p>
            <p className='text-sm opacity-90'>{err}</p>
          </div>
        </div>
        <Link href='/login' className='btn-primary'>
          {t("common.goToLogin") || "Go to login"}
        </Link>
      </main>
    );
  }

  return (
    <main className='max-w-3xl mx-auto px-4 py-6'>
      {/* Header */}
      <header className='flex items-center gap-3 mb-4'>
        <Link href='/' className='btn-secondary inline-flex items-center gap-2'>
          <ArrowLeft className='w-4 h-4' />
          {t("common.home") || "Home"}
        </Link>
        <h1 className='text-2xl font-bold ml-1'>
          {t("profile.title") || "My Profile"}
        </h1>
        <div className='ml-auto'>
          <button
            onClick={() => signOut({ callbackUrl: "/login" })}
            className='btn-secondary inline-flex items-center gap-2'
          >
            <LogOut className='w-4 h-4' />
            {t("nav.signOut")}
          </button>
        </div>
      </header>

      {/* Feedback */}
      {msg && (
        <div className='alert alert-success mb-4'>
          <CheckCircle2 />
          <div>
            <p className='font-semibold'>{t("common.success")}</p>
            <p className='text-sm opacity-90'>{msg}</p>
          </div>
        </div>
      )}
      {err && status === "authenticated" && (
        <div className='alert alert-error mb-4'>
          <AlertCircle />
          <div>
            <p className='font-semibold'>{t("common.error")}</p>
            <p className='text-sm opacity-90'>{err}</p>
          </div>
        </div>
      )}

      {/* Profile card */}
      <section className='card p-5'>
        <h2 className='text-lg font-semibold mb-3'>
          {t("profile.profile") || "Profile"}
        </h2>
        <form onSubmit={onSaveProfile} className='grid gap-4 max-w-xl'>
          <div>
            <label className='block text-sm font-medium mb-1'>
              {t("common.name")}
            </label>
            <input
              value={name}
              onChange={(e) => setName(e.target.value)}
              placeholder={t("common.yourName") || "Your name"}
              className='input-field'
            />
          </div>
          <div>
            <label className='block text-sm font-medium mb-1'>
              {t("login.email")}
            </label>
            <input
              type='email'
              value={email}
              onChange={(e) => setEmail(e.target.value)}
              className='input-field'
            />
          </div>
          <div className='pt-1'>
            <button
              type='submit'
              disabled={savingProfile}
              className='btn-primary inline-flex items-center gap-2 disabled:opacity-60'
            >
              <Save className='w-4 h-4' />
              {savingProfile
                ? t("common.saving") + "…"
                : t("profile.saveChanges") || "Save changes"}
            </button>
          </div>
        </form>
      </section>

      {/* Password card */}
      <section className='card p-5 mt-6'>
        <h2 className='text-lg font-semibold mb-3'>
          {t("profile.changePassword") || "Change password"}
        </h2>
        <form onSubmit={onChangePassword} className='grid gap-4 max-w-xl'>
          <div>
            <label className='block text-sm font-medium mb-1'>
              {t("profile.currentPassword") || "Current password"}
            </label>
            <input
              type='password'
              value={currentPassword}
              onChange={(e) => setCurrentPassword(e.target.value)}
              minLength={6}
              className='input-field'
            />
          </div>
          <div>
            <label className='block text-sm font-medium mb-1'>
              {t("profile.newPasswordMin") || "New password (min 6)"}
            </label>
            <input
              type='password'
              value={newPassword}
              onChange={(e) => setNewPassword(e.target.value)}
              minLength={6}
              className='input-field'
            />
          </div>
          <div className='pt-1'>
            <button
              type='submit'
              disabled={savingPassword}
              className='btn-primary inline-flex items-center gap-2 disabled:opacity-60'
            >
              <Save className='w-4 h-4' />
              {savingPassword
                ? t("profile.updating") + "…"
                : t("profile.updatePassword") || "Update password"}
            </button>
          </div>
        </form>
      </section>
    </main>
  );
}
